<?php
include './config.php';

// Database credentials
$dbname = $database_name;

// Create connection
$conn = new mysqli($servername, $username, $password, $dbname);

// Check connection
if ($conn->connect_error) {
    die(json_encode(["error" => "Connection failed: " . $conn->connect_error]));
}

// Retrieve parameters from the request
$selectedLocations = isset($_GET['locationValues']) ? json_decode($_GET['locationValues'], true) : [];
$selectedCategories = isset($_GET['categoryValues']) ? json_decode($_GET['categoryValues'], true) : [];
$selectedDegrees = isset($_GET['degreeValues']) ? json_decode($_GET['degreeValues'], true) : [];
$searchTerm = isset($_GET['searchTerm']) ? $_GET['searchTerm'] : '';
// $searchTermLocation = isset($_GET['searchLocation']) ? $_GET['searchLocation'] : '';


$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 30;
$offset = ($page - 1) * $limit;


// Initialize SQL query for jobs
$sql = "SELECT * FROM $table_name_oracle WHERE 1=1";
$params = [];
$types = '';

// Check if any locations are provided
if (!empty($selectedLocations)) {
    $locationConditions = [];

    foreach ($selectedLocations as $location) {
        // Normalize the input: lowercase and remove commas
        $normalizedLocation = strtolower(str_replace(',', '', $location));
        
        // SQL: normalize the DB column similarly for comparison
        $locationConditions[] = "LOWER(REPLACE(primaryLocation, ',', '')) LIKE ?";
        
        // Use wildcards for partial match if needed
        $params[] = "%$normalizedLocation%";
        $types .= 's';
    }

    $sql .= " AND (" . implode(' OR ', $locationConditions) . ")";
}

if (!empty($selectedCategories)) {
    $categoriesPlaceholder = implode(',', array_fill(0, count($selectedCategories), '?'));
    $sql .= " AND category IN ($categoriesPlaceholder)";
    $params = array_merge($params, $selectedCategories);
    $types .= str_repeat('s', count($selectedCategories));
} 

if (!empty($selectedDegrees)) {
    $degreesPlaceholder = implode(',', array_fill(0, count($selectedDegrees), '?'));
    $sql .= " AND studyLevel IN ($degreesPlaceholder)";
    $params = array_merge($params, $selectedDegrees);
    $types .= str_repeat('s', count($selectedDegrees));
}
$searchTerm = trim($searchTerm);
$searchTerm = preg_replace('/\s+/', ' ', $searchTerm);

// Step 1: Check if the search term matches any existing category
$categoryQuery = "SELECT category FROM $table_name_oracle WHERE category = ?";
$categoryStmt = $conn->prepare($categoryQuery);
$categoryStmt->bind_param('s', $searchTerm);
$categoryStmt->execute();
$categoryResult = $categoryStmt->get_result();

if ($categoryResult && $categoryResult->num_rows > 0) {
    // If the search term exactly matches a category, get jobs for that category
    $sql .= " AND category = ?";
    $params[] = $searchTerm;
    $types .= 's';
} else {
    // Step 2: If no exact match for category, handle normal search
    if (!empty($searchTerm)) {
        $searchWords = preg_split('/[\s, -&]+/', $searchTerm);
        $sql .= " AND ("; 
        $wordConditions = [];
        foreach ($searchWords as $word) {
            $word = trim($word);
            if (!empty($word)) {
                $searchParam = "%$word%";
                $wordConditions[] = "(title LIKE ? OR corporateDescriptionStr LIKE ? OR externalDescriptionStr LIKE ? OR primaryLocation LIKE ?)";
                $params[] = $searchParam; // title
                $params[] = $searchParam; // corporate description
                $params[] = $searchParam; // external description
                $params[] = $searchParam; // location
                $types .= 'ssss';
            }
        }
        $sql .= implode(' OR ', $wordConditions) . ")";
    }
}

// --- COUNT QUERY (before adding LIMIT/OFFSET) ---
$countSql = "SELECT COUNT(*) as total FROM $table_name_oracle WHERE 1=1";
$countSql .= substr($sql, strlen("SELECT * FROM $table_name_oracle WHERE 1=1"));
$countParams = $params;
$countTypes = $types;

// --- Add Pagination ---
$sql .= " LIMIT ? OFFSET ?";
$params[] = $limit;
$params[] = $offset;
$types .= 'ii';

// Prepare and execute the statement for jobs
$stmt = $conn->prepare($sql);
if ($stmt === false) {
    die(json_encode(["error" => "Prepare failed: " . $conn->error]));
}

$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
$jobs = $result->fetch_all(MYSQLI_ASSOC);   


if (!$result) {
    die(json_encode(["error" => "Query failed: " . $stmt->error]));
}


// --- Prepare and Execute Count Query ---
$countStmt = $conn->prepare($countSql);
if ($countStmt === false) {
    die(json_encode(["error" => "Count prepare failed: " . $conn->error]));
}
if (!empty($countParams)) {
    $countStmt->bind_param($countTypes, ...$countParams);
}
$countStmt->execute();
$countResult = $countStmt->get_result();
$totalRow = $countResult->fetch_assoc();
$totalCount = $totalRow['total'] ?? 0;


// Free result set
$result->free();

// Queries to get all unique categories, locations, and degree levels
$categoriesQuery = "SELECT DISTINCT category FROM $table_name_oracle";
$locationsQuery = "SELECT DISTINCT primaryLocation FROM $table_name_oracle";
$degreeLevelsQuery = "SELECT DISTINCT studyLevel FROM $table_name_oracle";

$categoriesResult = $conn->query($categoriesQuery);
$locationsResult = $conn->query($locationsQuery);
$degreeLevelsResult = $conn->query($degreeLevelsQuery);


if (!$categoriesResult || !$locationsResult || !$degreeLevelsResult) {
    die(json_encode(["error" => "Query failed: " . $conn->error]));
}

$categories = [];
$locations = [];
$degreeLevels = [];

while ($row = $categoriesResult->fetch_assoc()) {
    if (!empty($row['category'])) {
        $categories[] = $row['category'];
    }
}

while ($row = $locationsResult->fetch_assoc()) {
    if (!empty($row['primaryLocation'])) {
        $locations[] = $row['primaryLocation'];
    }
}

while ($row = $degreeLevelsResult->fetch_assoc()) {
    if (!empty($row['studyLevel'])) {
        $degreeLevels[] = $row['studyLevel'];
    }
}

// Free result sets
$categoriesResult->free();
$locationsResult->free();
$degreeLevelsResult->free();
$conn->close();

// Return success message
echo json_encode([
    "status" => "success",
    "count" => count($jobs), 
     "totalCount" => $totalCount,
    "currentPage" => $page,
    "limit" => $limit,
    "from" => "oracle",
    "message" => "Data retrieved successfully",
    "jobs" => $jobs,
    // "categories" => $categories,
    // "locations" => $locations,
    // "degreeLevels" => $degreeLevels
]);
?>
